-- client/brokers.lua (drop-in)
-- Spawns Fixer NPCs, opens dialog, and wires Accept/Decline to server.
-- Requires: ox_lib, ox_target. Works with ESX or standalone (server decides).

local D = {
  title      = 'FIXER',
  greet      = 'Need work?',
  intel      = 'Contract available. Intel attached.',
  accept     = 'Accept Contract',
  decline    = 'Decline',
  busy       = 'Busy — finish your current contract.',
  cooldown   = 'You are on cooldown',
  distance   = 'Distance',
  region     = 'Region',
}

local spawned = {}
local targets = {}

-- Utils
local function loadModel(model)
  if type(model) == 'string' then model = joaat(model) end
  if not IsModelValid(model) then return false end
  if not HasModelLoaded(model) then
    RequestModel(model)
    local t = GetGameTimer()
    while not HasModelLoaded(model) do
      Wait(10)
      if GetGameTimer() - t > 5000 then return false end
    end
  end
  return true
end

local function zoneLabelAt(coords)
  local zone = GetNameOfZone(coords.x, coords.y, coords.z)
  local label = zone and GetLabelText(zone) or nil
  if label and label ~= 'NULL' then return label end
  -- fallback to street name
  local s1 = GetStreetNameAtCoord(coords.x, coords.y, coords.z)
  return s1 and GetStreetNameFromHashKey(s1) or 'San Andreas'
end

local function groundVec3(x, y, z)
  local ok, gz = GetGroundZFor_3dCoord(x + 0.0, y + 0.0, (z or 100.0), false)
  return vector3(x, y, ok and gz or (z or 40.0))
end

-- Build a pre-plan for the briefing UI (region + hint + distance)
local function buildPrePlan()
  local me = PlayerPedId()
  local p = GetEntityCoords(me)

  -- choose a point 800–2200m away in a random direction
  local dist = math.random(800, 2200) + 0.0
  local ang = math.random() * math.pi * 2.0
  local dx, dy = math.cos(ang) * dist, math.sin(ang) * dist
  local pos = groundVec3(p.x + dx, p.y + dy, p.z + 30.0)

  local hint = pos + vector3(math.random(-40, 40) * 1.0, math.random(-40, 40) * 1.0, 0.0)
  hint = groundVec3(hint.x, hint.y, hint.z + 10.0)

  local plan = {
    pos      = pos,
    hdg      = math.random() * 360.0,
    hint     = hint,
    region   = zoneLabelAt(pos),
    distance = #(p - hint),
  }
  return plan
end

-- Open the broker dialog (ox_lib context)
local function openBrokerDialog(broker)
  -- Ask server if we can request (no active, no cooldown)
  local gate = lib.callback.await('hitman:server:canRequest', false)
  if not gate or not gate.ok then
    local msg = (gate and gate.reason == 'active') and D.busy
              or (('%s%s'):format(D.cooldown, (gate and gate.left and (' ('..math.ceil(gate.left)..'s)') or '')))
    lib.notify({ description = msg, type = 'error' })
    return
  end

  local plan = buildPrePlan()

  local prettyDist = ('%dm'):format(math.floor(plan.distance + 0.5))
  local desc = ('%s\n\n• %s: %s\n• %s: %s')
      :format(D.intel, D.region, plan.region, D.distance, prettyDist)

  lib.registerContext({
    id = 'hitman_broker_context',
    title = D.title,
    description = desc,
    options = {
      {
        title = D.accept,
        icon = 'fa-solid fa-briefcase',
        onSelect = function()
          -- hand the pre-plan to mission logic, then tell server to start
          TriggerEvent('hitman:client:setPrePlan', plan)
          TriggerServerEvent('hitman:server:acceptContract')
        end
      },
      {
        title = D.decline,
        icon = 'fa-solid fa-xmark',
        onSelect = function()
          -- optional: tell server we declined to clear any stray state
          TriggerServerEvent('hitman:server:decline')
        end
      }
    }
  })

  lib.showContext('hitman_broker_context')
end

-- Spawn brokers from Config.Brokers
CreateThread(function()
  for i, b in ipairs(Config.Brokers or {}) do
    if loadModel(b.model or `cs_movpremmale`) then
      local ped = CreatePed(4, (type(b.model)=='string' and joaat(b.model) or b.model),
        b.coords.x, b.coords.y, b.coords.z - 1.0, b.heading or 0.0, false, true)
      SetEntityInvincible(ped, true)
      SetBlockingOfNonTemporaryEvents(ped, true)
      FreezeEntityPosition(ped, true)
      if b.scenario then
        TaskStartScenarioInPlace(ped, b.scenario, 0, true)
      end

      -- Optional blip
      if b.blip then
        local blip = AddBlipForCoord(b.coords.x, b.coords.y, b.coords.z)
        SetBlipSprite(blip, b.blip.sprite or 280)
        SetBlipColour(blip, b.blip.color or 1)
        SetBlipScale(blip, b.blip.scale or 0.8)
        SetBlipAsShortRange(blip, true)
        BeginTextCommandSetBlipName('STRING')
        AddTextComponentString(b.blip.name or 'Fixer')
        EndTextCommandSetBlipName(blip)
      end

      -- ox_target
      exports.ox_target:addLocalEntity(ped, {{
        name = ('hitman_broker_%s'):format(i),
        icon = 'fa-solid fa-user-tie',
        label = b.label or 'Talk to Fixer',
        distance = b.distance or 2.0,
        onSelect = function() openBrokerDialog(b) end
      }})

      spawned[#spawned+1] = ped
    end
    Wait(75)
  end
end)

-- Cleanup on resource stop
AddEventHandler('onResourceStop', function(res)
  if res ~= GetCurrentResourceName() then return end
  for _, ped in ipairs(spawned) do
    if DoesEntityExist(ped) then
      pcall(function() exports.ox_target:removeLocalEntity(ped) end)
      DeleteEntity(ped)
    end
  end
end)
