-- qb-parkour / client/cl_parkour.lua
local QBCore = exports['qb-core']:GetCoreObject()

local enabled = Config.ParkControlsEnabledByDefault
local lastUse = {}
local chainOpenUntil = 0

-- simple helper
local function now()
    return GetGameTimer()
end

local function canUse(move)
    local t = now()
    local lu = lastUse[move] or 0
    return (t - lu) >= Config:GetCooldown(move)
end

local function used(move)
    lastUse[move] = now()
    chainOpenUntil = now() + Config.ChainWindowMs
end

local function notify(msg, type)
    if Config.ShowCooldownNotifications then
        QBCore.Functions.Notify(msg, type or 'primary', 2000)
    end
end

-- animation utils
local function loadDict(dict)
    if not HasAnimDictLoaded(dict) then
        RequestAnimDict(dict)
        while not HasAnimDictLoaded(dict) do
            Wait(5)
        end
    end
end

local function playOnce(ped, dict, anim, blendIn, blendOut, dur, flag, playbackRate)
    loadDict(dict)
    TaskPlayAnim(ped, dict, anim, blendIn or 3.0, blendOut or 1.0, dur or -1, flag or 0, playbackRate or 0, false, false, false)
end

-- physics helpers
local function pushForward(ped, x, y, z)
    ApplyForceToEntityCenterOfMass(ped, 1, x or 0.0, y or 4.0, z or 0.6, true, true, true, true)
end

-- graceful recover
local function clear(ped)
    ClearPedSecondaryTask(ped)
end

-- Moves (freestyle-minded)
local function move_ForwardLeap(ped)
    playOnce(ped, 'move_climb', 'clamber_to_jump', 3.0, 1.0, 1200, 0, 0)
    Wait(120)
    pushForward(ped, 0.0, Config.Force.Medium, 0.8)
    Wait(250)
    pushForward(ped, 0.0, Config.Force.Small, 0.4)
    used('ForwardLeap')
end

local function move_ForwardRoll(ped)
    playOnce(ped, 'move_fall', 'land_roll', 3.0, 1.0, 900, 0, 0)
    Wait(50)
    pushForward(ped, 0.0, Config.Force.Small, 0.0)
    used('ForwardRoll')
end

local function move_ForwardSlide(ped)
    playOnce(ped, 'missheistfbi3b_ig6_v2', 'rubble_slide_gunman', 3.0, 1.0, 900, 0, 0)
    Wait(80)
    pushForward(ped, 0.0, Config.Force.Medium + 1.0, 0.0)
    used('ForwardSlide')
end

local function move_ForwardDive(ped)
    playOnce(ped, 'move_climb', 'clamberpose_to_dive', 3.0, 1.0, 1000, 0, 0)
    Wait(320)
    pushForward(ped, 0.0, Config.Force.Large, 0.8)
    Wait(320)
    playOnce(ped, 'move_fall', 'land_roll', 3.0, 1.0, 800, 0, 0)
    used('ForwardDive')
end

local function move_BackFlip(ped)
    playOnce(ped, 'anim@arena@celeb@flat@solo@no_props@', 'flip_a_player_a', 3.0, 1.0, 2000, 0, 0)
    used('BackFlip')
end

local function move_ForceClimb(ped)
    TaskClimb(ped, false)
    Wait(100)
    pushForward(ped, 0.0, 0.0, Config.Force.Medium)
    used('ForceClimb')
end

local function move_OverJump(ped)
    playOnce(ped, 'missfra0_chop_fchase', 'ballasog_carbonnetslide', 3.0, 1.0, 900, 0, 0)
    Wait(150)
    pushForward(ped, 0.0, Config.Force.Medium, 0.9)
    used('OverJump')
end

local function move_LeftLongDive(ped)
    playOnce(ped, 'mini@tennis', 'dive_bh_long_lo', 3.0, 1.0, 1200, 0, 0)
    Wait(100)
    ApplyForceToEntityCenterOfMass(ped, 1, -Config.Force.Small, Config.Force.Small, 0.6, true, true, true, true)
    used('LeftLongDive')
end

local function move_ShortLeftDive(ped)
    playOnce(ped, 'mini@tennis', 'dive_bh_short_lo', 3.0, 1.0, 900, 0, 0)
    Wait(80)
    ApplyForceToEntityCenterOfMass(ped, 1, -Config.Force.Small, Config.Force.Small, 0.2, true, true, true, true)
    used('ShortLeftDive')
end

local function move_LongRightDive(ped)
    playOnce(ped, 'mini@tennis', 'dive_fh_long_hi', 3.0, 1.0, 1200, 0, 0)
    Wait(100)
    ApplyForceToEntityCenterOfMass(ped, 1, Config.Force.Small, Config.Force.Small, 0.6, true, true, true, true)
    used('LongRightDive')
end

local function move_ShortRightDive(ped)
    playOnce(ped, 'mini@tennis', 'dive_fh_short_lo', 3.0, 1.0, 900, 0, 0)
    Wait(80)
    ApplyForceToEntityCenterOfMass(ped, 1, Config.Force.Small, Config.Force.Small, 0.2, true, true, true, true)
    used('ShortRightDive')
end

local function move_BackwardDive(ped)
    playOnce(ped, 'move_avoidance@generic_m', 'react_right_side_dive_back', 3.0, 1.0, 900, 0, 0)
    used('BackwardDive')
end

-- map of move handlers
local Moves = {
    ForwardLeap = move_ForwardLeap,
    ForwardRoll = move_ForwardRoll,
    ForwardSlide = move_ForwardSlide,
    ForwardDive = move_ForwardDive,
    BackFlip = move_BackFlip,
    ForceClimb = move_ForceClimb,
    OverJump = move_OverJump,
    LeftLongDive = move_LeftLongDive,
    ShortLeftDive = move_ShortLeftDive,
    LongRightDive = move_LongRightDive,
    ShortRightDive = move_ShortRightDive,
    BackwardDive = move_BackwardDive,
}

-- Core loop (single lightweight tick)
CreateThread(function()
    while true do
        local sleep = 250
        if enabled then
            sleep = 0
            local ped = PlayerPedId()
            if DoesEntityExist(ped) and not IsEntityDead(ped) and IsPedOnFoot(ped) and not IsPedRagdoll(ped) then
                -- read passive key in either device
                local pass = (IsControlPressed(Config.Keys.PassiveKey.kbmGroup, Config.Keys.PassiveKey.kbm)
                           or IsControlPressed(Config.Keys.PassiveKey.padGroup, Config.Keys.PassiveKey.pad))

                if pass then
                    -- check all configured moves
                    for moveKey, moveCfg in pairs(Config.Keys) do
                        if moveKey ~= 'PassiveKey' then
                            if (IsControlPressed(moveCfg.kbmGroup, moveCfg.kbm) or IsControlPressed(moveCfg.padGroup, moveCfg.pad)) then
                                if canUse(moveKey) then
                                    local fn = Moves[moveKey]
                                    if fn then
                                        fn(ped)
                                    end
                                end
                            end
                        end
                    end
                end
            end
        end
        Wait(sleep)
    end
end)

-- Toggle from server command
RegisterNetEvent('qb-parkour:client:toggle', function(state)
    if state == nil then enabled = not enabled else enabled = state end
    if enabled then
        notify('Parkour activé (freestyle).', 'success')
    else
        notify('Parkour désactivé.', 'error')
    end
end)

RegisterNetEvent('qb-parkour:client:notifytoggle', function()
    Config.ShowCooldownNotifications = not Config.ShowCooldownNotifications
    notify(('Notifications: %s'):format(Config.ShowCooldownNotifications and 'ON' or 'OFF'), 'primary')
end)

-- Helpful export for other scripts
exports('SetEnabled', function(state)
    enabled = state and true or false
end)
